<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if(!class_exists('\Twig\Environment')){
	require_once(__DIR__ . '/../../twig-vendor/autoload.php');
}

class Softlite_Dynamic_List extends \Elementor\Widget_Base {

	public function get_name() {
		return 'softlite_dynamic_list';
	}

	public function get_title() {
		return esc_html__( 'Dynamic List', 'softlite' );
	}

	public function get_icon() {
		return 'eicon-post-list';
	}

	public function get_categories() {
		return [ 'softlite' ];
	}

	public function get_keywords() {
		return [ 'icon', 'icon list', 'softlite', 'dynamic' ];
	}

	public function get_style_depends() {
		return [
			'swiper',
			'e-swiper',
			'softlite-integration-widget-style'
		];
	}

	public function get_script_depends() {
		wp_register_script(
            "swiper",
            ELEMENTOR_ASSETS_URL . "/lib/swiper/v8/swiper.min.js",
            ["jquery"],
            false,
            true
        );
		return [ 
			'swiper',
			'softlite-elementor-widgets-script'
		];
	}

	public function has_widget_inner_wrapper(): bool {
		return ! Elementor\Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
	}

	protected function register_controls() {
		$this->start_controls_section(
			'section_icon',
			[
				'label' => esc_html__( 'Dynamic List', 'elementor' ),
			]
		);

		$repeater = new Elementor\Repeater();

		$repeater->add_control(
			'selected_icon_enable',
			[
				'label' => esc_html__( 'Icon/Image Enable', 'elementor' ),
				'type' => Elementor\Controls_Manager::HIDDEN,
			]
		);

		$repeater->add_control(
			'selected_icon_source',
			[
				'label' => esc_html__( 'Icon Source', 'elementor' ),
				'type' => Elementor\Controls_Manager::SELECT,
				'options' => [
					'icon' => esc_html__( 'Icon library', 'elementor' ),
					'image' => esc_html__( 'Image', 'elementor' ),
				],
				'default' => 'icon',
				'condition' => [
					'selected_icon_enable' => '1',
				],
			]
		);

		$repeater->add_control(
			'selected_icon',
			[
				'label' => esc_html__( 'Icon', 'elementor' ),
				'type' => Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default' => [
					'value' => 'fas fa-star',
					'library' => 'fa-solid',
				],
				'condition' => [
					'selected_icon_enable' => '1',
					'selected_icon_source' => 'icon',
				],
			]
		);

        $repeater->add_control(
			'selected_icon_image_source',
			[
				'label' => esc_html__( 'Image Source', 'elementor' ),
				'type' => Elementor\Controls_Manager::SELECT,
				'options' => [
					'upload' => esc_html__( 'Upload File', 'elementor' ),
					'external' => esc_html__( 'External URL', 'elementor' ),
				],
				'default' => 'upload',
                'condition' => [
					'selected_icon_enable' => '1',
					'selected_icon_source' => 'image',
				],
			]
		);

        $repeater->add_control(
            'selected_icon_image',
            [
                'label' => esc_html__( 'Image', 'elementor' ),
                'type' => Elementor\Controls_Manager::MEDIA,
                'dynamic' => [
                    'active' => true,
                ],
                'condition' => [
					'selected_icon_enable' => '1',
                    'selected_icon_source' => 'image',
                    'selected_icon_image_source' => 'upload',
                ],
            ]
        );

        $repeater->add_control(
			'selected_icon_image_external_url' ,
			[
				'label' => esc_html__( 'Image URL', 'elementor' ),
				'type' => Elementor\Controls_Manager::TEXT,
				'ai' => [
					'active' => false,
				],
				'condition' => [
					'selected_icon_enable' => '1',
                    'selected_icon_source' => 'image',
					'selected_icon_image_source' => 'external',
				],
			]
		);

		$repeater->add_control(
			'text_1_enable',
			[
				'label' => esc_html__( 'Text 1 Enable', 'elementor' ),
				'type' => Elementor\Controls_Manager::HIDDEN,
			]
		);

		$repeater->add_control(
			'text_1',
			[
				'label' => esc_html__( 'Text 1', 'elementor' ),
				'type' => Elementor\Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'List Item', 'elementor' ),
				'default' => esc_html__( 'List Item', 'elementor' ),
				'dynamic' => [
					'active' => true,
				],
				'condition' => [
					'text_1_enable' => '1',
				],
			]
		);

		$repeater->add_control(
			'text_2_enable',
			[
				'label' => esc_html__( 'Text 2 Enable', 'elementor' ),
				'type' => Elementor\Controls_Manager::HIDDEN,
			]
		);

		$repeater->add_control(
			'text_2',
			[
				'label' => esc_html__( 'Text 2', 'elementor' ),
				'type' => Elementor\Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'List Item', 'elementor' ),
				'default' => esc_html__( 'List Item', 'elementor' ),
				'dynamic' => [
					'active' => true,
				],
				'condition' => [
					'text_2_enable' => '1',
				],
			]
		);

		$repeater->add_control(
			'text_3_enable',
			[
				'label' => esc_html__( 'Text 3 Enable', 'elementor' ),
				'type' => Elementor\Controls_Manager::HIDDEN,
			]
		);

		$repeater->add_control(
			'text_3',
			[
				'label' => esc_html__( 'Text 3', 'elementor' ),
				'type' => Elementor\Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'List Item', 'elementor' ),
				'default' => esc_html__( 'List Item', 'elementor' ),
				'dynamic' => [
					'active' => true,
				],
				'condition' => [
					'text_3_enable' => '1',
				],
			]
		);

		$repeater->add_control(
			'button_text_enable',
			[
				'label' => esc_html__( 'Button Text Enable', 'elementor' ),
				'type' => Elementor\Controls_Manager::HIDDEN,
			]
		);

        $repeater->add_control(
			'button_text',
			[
				'label' => esc_html__( 'Button Text', 'elementor' ),
                'label_block' => true,
				'type' => Elementor\Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'condition' => [
					'button_text_enable' => '1',
				],
			]
		);

		$repeater->add_control(
			'link',
			[
				'label' => esc_html__( 'Link', 'elementor' ),
				'type' => Elementor\Controls_Manager::URL,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'dynamic_list',
			[
				'label' => esc_html__( 'Items', 'elementor' ),
				'type' => Elementor\Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'text_1' => esc_html__( 'List Item #1', 'elementor' ),
					],
					[
						'text_1' => esc_html__( 'List Item #2', 'elementor' ),
					],
					[
						'text_1' => esc_html__( 'List Item #3', 'elementor' ),
					],
				],
				'title_field' => '{{{ text_1 }}}',
			]
		);

		$this->add_control(
			'dynamic_template',
            [
                'label' => esc_html__( 'Dynamic Template', 'elementor' ),
                'type' => Elementor\Controls_Manager::HIDDEN,
                'language' => 'html',
                'rows' => 20,
                'default' => '<div class="softlite-dynamic-list-wrapper">{{settings.text_1}}</div>',
                'ai' => [
					'active' => false,
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_additional_options',
			[
				'label' => esc_html__( 'Additional Options', 'elementor' ),
			]
		);

		$this->add_control(
			'slider',
			[
				'label' => __( 'Enable Slider', 'pafe' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'label_off',
				'label_on' => 'Yes',
				'label_off' => 'No',
				'return_value' => 'yes',
			]
		);

		$this->add_responsive_control(
			'slider_slides_to_show',
			[
				'label' => __( 'Slides to Show', 'pafe' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 1,
				'condition' => [
					'slider' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'slider_slides_to_scroll',
			[
				'label' => __( 'Slides to Scroll', 'pafe' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 1,
				'condition' => [
					'slider' => 'yes',
				],
			]
		);

		$this->add_control(
			'navigation',
			[
				'label' => esc_html__( 'Navigation', 'elementor' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'none',
				'options' => [
					'both' => esc_html__( 'Arrows and Dots', 'elementor' ),
					'arrows' => esc_html__( 'Arrows', 'elementor' ),
					'dots' => esc_html__( 'Dots', 'elementor' ),
					'none' => esc_html__( 'None', 'elementor' ),
				],
				'frontend_available' => true,
				'condition' => [
					'slider' => 'yes',
				],
			]
		);

		$this->add_control(
			'navigation_previous_icon',
			[
				'label' => esc_html__( 'Previous Arrow Icon', 'elementor' ),
				'type' => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'skin' => 'inline',
				'label_block' => false,
				'skin_settings' => [
					'inline' => [
						'none' => [
							'label' => 'Default',
							'icon' => 'eicon-chevron-left',
						],
						'icon' => [
							'icon' => 'eicon-star',
						],
					],
				],
				'recommended' => [
					'fa-regular' => [
						'arrow-alt-circle-left',
						'caret-square-left',
					],
					'fa-solid' => [
						'angle-double-left',
						'angle-left',
						'arrow-alt-circle-left',
						'arrow-circle-left',
						'arrow-left',
						'caret-left',
						'caret-square-left',
						'chevron-circle-left',
						'chevron-left',
						'long-arrow-alt-left',
					],
				],
				'conditions' => [
					'relation' => 'or',
					'terms' => [
						[
							'name' => 'navigation',
							'operator' => '=',
							'value' => 'both',
						],
						[
							'name' => 'navigation',
							'operator' => '=',
							'value' => 'arrows',
						],
					],
				],
			]
		);

		$this->add_control(
			'navigation_next_icon',
			[
				'label' => esc_html__( 'Next Arrow Icon', 'elementor' ),
				'type' => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'skin' => 'inline',
				'label_block' => false,
				'skin_settings' => [
					'inline' => [
						'none' => [
							'label' => 'Default',
							'icon' => 'eicon-chevron-right',
						],
						'icon' => [
							'icon' => 'eicon-star',
						],
					],
				],
				'recommended' => [
					'fa-regular' => [
						'arrow-alt-circle-right',
						'caret-square-right',
					],
					'fa-solid' => [
						'angle-double-right',
						'angle-right',
						'arrow-alt-circle-right',
						'arrow-circle-right',
						'arrow-right',
						'caret-right',
						'caret-square-right',
						'chevron-circle-right',
						'chevron-right',
						'long-arrow-alt-right',
					],
				],
				'conditions' => [
					'relation' => 'or',
					'terms' => [
						[
							'name' => 'navigation',
							'operator' => '=',
							'value' => 'both',
						],
						[
							'name' => 'navigation',
							'operator' => '=',
							'value' => 'arrows',
						],
					],
				],
			]
		);

		$this->add_control(
			'autoplay',
			[
				'label' => esc_html__( 'Autoplay', 'elementor' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'elementor' ),
				'label_off' => esc_html__( 'No', 'elementor' ),
				'return_value' => 'yes',
				'default' => '',
				'frontend_available' => true,
				'condition' => [
					'slider' => 'yes',
				],
			]
		);

		// $this->add_control(
		// 	'pause_on_hover',
		// 	[
		// 		'label' => esc_html__( 'Pause on Hover', 'elementor' ),
		// 		'type' => \Elementor\Controls_Manager::SWITCHER,
		// 		'label_on' => esc_html__( 'Yes', 'elementor' ),
		// 		'label_off' => esc_html__( 'No', 'elementor' ),
		// 		'return_value' => 'yes',
		// 		'default' => 'yes',
		// 		'condition' => [
		// 			'autoplay' => 'yes',
		// 		],
		// 		'render_type' => 'none',
		// 		'frontend_available' => true,
		// 	]
		// );

		$this->add_control(
			'pause_on_interaction',
			[
				'label' => esc_html__( 'Pause on Interaction', 'elementor' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'elementor' ),
				'label_off' => esc_html__( 'No', 'elementor' ),
				'return_value' => 'yes',
				'default' => 'yes',
				'condition' => [
					'autoplay' => 'yes',
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'autoplay_speed',
			[
				'label' => esc_html__( 'Autoplay Speed', 'elementor' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 5000,
				'condition' => [
					'autoplay' => 'yes',
				],
				'render_type' => 'none',
				'frontend_available' => true,
			]
		);

		// Loop requires a re-render so no 'render_type = none'
		$this->add_control(
			'infinite',
			[
				'label' => esc_html__( 'Infinite Loop', 'elementor' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'elementor' ),
				'label_off' => esc_html__( 'No', 'elementor' ),
				'return_value' => 'yes',
				'default' => '',
				'frontend_available' => true,
				'condition' => [
					'slider' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_dynamic_list',
			[
				'label' => esc_html__( 'List', 'elementor' ),
				'tab' => Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'flex_direction',
			[
				'label' => esc_html__( 'Flex Direction', 'elementor' ),
				'type' => Elementor\Controls_Manager::SELECT,
				'options' => [
					'row' => esc_html__( 'Row', 'elementor' ),
					'column' => esc_html__( 'Column', 'elementor' ),
					'row-reverse' => esc_html__( 'Row Reverse', 'elementor' ),
					'column-reverse' => esc_html__( 'Column Reverse', 'elementor' ),
				],
				'default' => 'row',
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list' => 'flex-direction: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'flex_wrap',
			[
				'label' => esc_html__( 'Flex Wrap', 'elementor' ),
				'type' => Elementor\Controls_Manager::SELECT,
				'options' => [
					'wrap' => esc_html__( 'Wrap', 'elementor' ),
					'nowrap' => esc_html__( 'No Wrap', 'elementor' ),
				],
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list' => 'flex-wrap: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'column_gap',
			[
				'label' => esc_html__( 'Column Gap', 'softlite' ),
				'type' => Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', 'rem', 'custom' ],
				'range' => [
					'px' => [
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list' => 'column-gap: {{SIZE}}{{UNIT}}',
				],
				'render_type' => 'template',
			]
		);

		$this->add_responsive_control(
			'row_gap',
			[
				'label' => esc_html__( 'Row Gap', 'softlite' ),
				'type' => Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', 'rem', 'custom' ],
				'range' => [
					'px' => [
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list' => 'row-gap: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'item_width',
			[
				'label' => esc_html__( 'Item Width', 'elementor' ),
				'type' => Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'em', 'rem', 'vw', 'custom' ],
				'range' => [
					'px' => [
						'min' => 6,
					],
					'%' => [
						'min' => 6,
					],
					'vw' => [
						'min' => 6,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list-item' => 'width: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->start_controls_tabs( 'tabs_style');

		$this->start_controls_tab(
			'tab_normal',
			[
				'label' => esc_html__( 'Normal', 'elementor' ),
			]
		);

		$this->add_group_control(
			Elementor\Group_Control_Background::get_type(),
			[
				'name' => 'background',
				'label' => esc_html__( 'Item Background', 'elementor' ),
				'types' => [ 'classic', 'gradient' ],
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .softlite-dynamic-list-item-inner',
				'fields_options' => [
					'background' => [
						'default' => 'classic',
					],
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_hover',
			[
				'label' => esc_html__( 'Hover', 'elementor' ),
			]
		);

		$this->add_group_control(
			Elementor\Group_Control_Background::get_type(),
			[
				'name' => 'background_hover',
				'label' => esc_html__( 'Item Background', 'elementor' ),
				'types' => [ 'classic', 'gradient' ],
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .softlite-dynamic-list-item-inner:hover, {{WRAPPER}} .softlite-dynamic-list-item-inner:focus',
				'fields_options' => [
					'background' => [
						'default' => 'classic',
					],
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_icon_style',
			[
				'label' => esc_html__( 'Icon/Image', 'elementor' ),
				'tab' => Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'icon_size',
			[
				'label' => esc_html__( 'Size', 'elementor' ),
				'type' => Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'em', 'rem', 'vw', 'custom' ],
				'default' => [
					'size' => 100,
					'unit' => '%',
				],
				'range' => [
					'px' => [
						'min' => 6,
						'max' => 1000,
					],
					'%' => [
						'min' => 6,
					],
					'vw' => [
						'min' => 6,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-icon' => 'font-size: {{SIZE}}{{UNIT}}; width: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .softlite-dynamic-icon svg' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs( 'icon_colors' );

		$this->start_controls_tab(
			'icon_colors_normal',
			[
				'label' => esc_html__( 'Normal', 'elementor' ),
			]
		);

		$this->add_control(
			'icon_color',
			[
				'label' => esc_html__( 'Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .softlite-dynamic-icon svg' => 'fill: {{VALUE}};',
				],
				'global' => [
					'default' => Elementor\Core\Kits\Documents\Tabs\Global_Colors::COLOR_PRIMARY,
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'icon_colors_hover',
			[
				'label' => esc_html__( 'Hover', 'elementor' ),
			]
		);

		$this->add_control(
			'icon_color_hover',
			[
				'label' => esc_html__( 'Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list-item-inner:hover .softlite-dynamic-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .softlite-dynamic-list-item-inner:hover .softlite-dynamic-icon svg' => 'fill: {{VALUE}};',
				],
			]
		);

		// $this->add_control(
		// 	'icon_color_hover_transition',
		// 	[
		// 		'label' => esc_html__( 'Transition Duration', 'elementor' ),
		// 		'type' => Elementor\Controls_Manager::SLIDER,
		// 		'size_units' => [ 's', 'ms', 'custom' ],
		// 		'default' => [
		// 			'unit' => 's',
		// 			'size' => 0.3,
		// 		],
		// 		'selectors' => [
		// 			'{{WRAPPER}} .softlite-dynamic-list-icon i' => 'transition: color {{SIZE}}{{UNIT}}',
		// 			'{{WRAPPER}} .softlite-dynamic-list-icon svg' => 'transition: fill {{SIZE}}{{UNIT}}',
		// 		],
		// 	]
		// );

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_text_style',
			[
				'label' => esc_html__( 'Text', 'elementor' ),
				'tab' => Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'text_1_typography',
				'label' => esc_html__( 'Text 1 Typography', 'elementor' ),
				'selector' => '{{WRAPPER}} .softlite-dynamic-list-item-text-1',
				'global' => [
					'default' => Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_TEXT,
				],
			]
		);

		$this->add_group_control(
			Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'text_2_typography',
				'label' => esc_html__( 'Text 2 Typography', 'elementor' ),
				'selector' => '{{WRAPPER}} .softlite-dynamic-list-item-text-2',
				'global' => [
					'default' => Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_TEXT,
				],
			]
		);

		$this->add_group_control(
			Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'text_3_typography',
				'label' => esc_html__( 'Text 3 Typography', 'elementor' ),
				'selector' => '{{WRAPPER}} .softlite-dynamic-list-item-text-3',
				'global' => [
					'default' => Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_TEXT,
				],
			]
		);

		$this->start_controls_tabs( 'text_colors' );

		$this->start_controls_tab(
			'text_colors_normal',
			[
				'label' => esc_html__( 'Normal', 'elementor' ),
			]
		);

		$this->add_control(
			'text_1_color',
			[
				'label' => esc_html__( 'Text 1 Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list-item-text-1' => 'color: {{VALUE}};',
				],
				'global' => [
					'default' => Elementor\Core\Kits\Documents\Tabs\Global_Colors::COLOR_SECONDARY,
				],
			]
		);

		$this->add_control(
			'text_2_color',
			[
				'label' => esc_html__( 'Text 2 Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list-item-text-2' => 'color: {{VALUE}};',
				],
				'global' => [
					'default' => Elementor\Core\Kits\Documents\Tabs\Global_Colors::COLOR_SECONDARY,
				],
			]
		);

		$this->add_control(
			'text_3_color',
			[
				'label' => esc_html__( 'Text 3 Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list-item-text-3' => 'color: {{VALUE}};',
				],
				'global' => [
					'default' => Elementor\Core\Kits\Documents\Tabs\Global_Colors::COLOR_SECONDARY,
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'text_colors_hover',
			[
				'label' => esc_html__( 'Hover', 'elementor' ),
			]
		);

		$this->add_control(
			'text_1_color_hover',
			[
				'label' => esc_html__( 'Text 1 Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list-item-inner:hover .softlite-dynamic-list-item-text-1' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'text_2_color_hover',
			[
				'label' => esc_html__( 'Text 2 Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list-item-inner:hover .softlite-dynamic-list-item-text-2' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'text_3_color_hover',
			[
				'label' => esc_html__( 'Text 3 Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list-item-inner:hover .softlite-dynamic-list-item-text-3' => 'color: {{VALUE}};',
				],
			]
		);

		// $this->add_control(
		// 	'text_color_hover_transition',
		// 	[
		// 		'label' => esc_html__( 'Transition Duration', 'elementor' ),
		// 		'type' => Elementor\Controls_Manager::SLIDER,
		// 		'size_units' => [ 's', 'ms', 'custom' ],
		// 		'default' => [
		// 			'unit' => 's',
		// 			'size' => 0.3,
		// 		],
		// 		'selectors' => [
		// 			'{{WRAPPER}} .softlite-dynamic-list-text' => 'transition: color {{SIZE}}{{UNIT}}',
		// 		],
		// 	]
		// );

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_button_style',
			[
				'label' => esc_html__( 'Button', 'elementor' ),
				'tab' => Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Elementor\Group_Control_Typography::get_type(),
			[
				'name' => 'button_text_typography',
				'label' => esc_html__( 'Button Text Typography', 'elementor' ),
				'global' => [
					'default' => Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_ACCENT,
				],
				'selector' => '{{WRAPPER}} .softlite-dynamic-list-item-button-text',
			]
		);

		$this->start_controls_tabs( 'tabs_button_style');

		$this->start_controls_tab(
			'tab_button_normal',
			[
				'label' => esc_html__( 'Normal', 'elementor' ),
			]
		);

		$this->add_control(
			'button_text_color',
			[
				'label' => esc_html__( 'Button Text Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list-item-button-text' => 'fill: {{VALUE}}; color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'button_background_color',
			[
				'label' => esc_html__( 'Button Background Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list-item-button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_button_hover',
			[
				'label' => esc_html__( 'Hover', 'elementor' ),
			]
		);

		$this->add_control(
			'button_text_color_hover',
			[
				'label' => esc_html__( 'Button Text Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list-item-inner:hover .softlite-dynamic-list-item-button-text, {{WRAPPER}} .softlite-dynamic-list-item-inner:focus .softlite-dynamic-list-item-button-text' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'button_background_color_hover',
			[
				'label' => esc_html__( 'Button Background Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .softlite-dynamic-list-item-inner:hover .softlite-dynamic-list-item-button, {{WRAPPER}}.softlite-dynamic-list-item-inner:focus .softlite-dynamic-list-item-button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_navigation',
			[
				'label' => esc_html__( 'Slider Navigation', 'elementor' ),
				'tab' => Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'navigation' => [ 'arrows', 'dots', 'both' ],
					'slider' => 'yes',
				],
			]
		);

		$this->add_control(
			'heading_style_arrows',
			[
				'label' => esc_html__( 'Arrows', 'elementor' ),
				'type' => Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'navigation' => [ 'arrows', 'both' ],
				],
			]
		);

		$this->add_control(
			'arrows_position',
			[
				'label' => esc_html__( 'Position', 'elementor' ),
				'type' => Elementor\Controls_Manager::SELECT,
				'default' => 'inside',
				'options' => [
					'inside' => esc_html__( 'Inside', 'elementor' ),
					'outside' => esc_html__( 'Outside', 'elementor' ),
				],
				'prefix_class' => 'elementor-arrows-position-',
				'condition' => [
					'navigation' => [ 'arrows', 'both' ],
				],
			]
		);

		$this->add_responsive_control(
			'arrows_size',
			[
				'label' => esc_html__( 'Size', 'elementor' ),
				'type' => Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', 'rem', 'custom' ],
				'range' => [
					'px' => [
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .elementor-swiper-button.elementor-swiper-button-prev, {{WRAPPER}} .elementor-swiper-button.elementor-swiper-button-next' => 'font-size: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'navigation' => [ 'arrows', 'both' ],
				],
			]
		);

		$this->add_control(
			'arrows_color',
			[
				'label' => esc_html__( 'Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-swiper-button.elementor-swiper-button-prev, {{WRAPPER}} .elementor-swiper-button.elementor-swiper-button-next' => 'color: {{VALUE}};',
					'{{WRAPPER}} .elementor-swiper-button.elementor-swiper-button-prev svg, {{WRAPPER}} .elementor-swiper-button.elementor-swiper-button-next svg' => 'fill: {{VALUE}};',
				],
				'condition' => [
					'navigation' => [ 'arrows', 'both' ],
				],
			]
		);

		$this->add_control(
			'heading_style_dots',
			[
				'label' => esc_html__( 'Pagination', 'elementor' ),
				'type' => Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'navigation' => [ 'dots', 'both' ],
				],
			]
		);

		$this->add_control(
			'dots_position',
			[
				'label' => esc_html__( 'Position', 'elementor' ),
				'type' => Elementor\Controls_Manager::SELECT,
				'default' => 'outside',
				'options' => [
					'outside' => esc_html__( 'Outside', 'elementor' ),
					'inside' => esc_html__( 'Inside', 'elementor' ),
				],
				'prefix_class' => 'elementor-pagination-position-',
				'condition' => [
					'navigation' => [ 'dots', 'both' ],
				],
			]
		);

		$this->add_responsive_control(
			'dots_gap',
			[
				'label' => esc_html__( 'Space Between Dots', 'elementor' ),
				'type' => Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', 'rem', 'custom' ],
				'range' => [
					'px' => [
						'max' => 20,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-bullet' => '--swiper-pagination-bullet-horizontal-gap: {{SIZE}}{{UNIT}}; --swiper-pagination-bullet-vertical-gap: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'navigation' => [ 'dots', 'both' ],
				],
			]
		);

		$this->add_responsive_control(
			'dots_size',
			[
				'label' => esc_html__( 'Size', 'elementor' ),
				'type' => Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', 'rem', 'custom' ],
				'range' => [
					'px' => [
						'max' => 20,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-bullet' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'navigation' => [ 'dots', 'both' ],
				],
			]
		);

		$this->add_control(
			'dots_inactive_color',
			[
				'label' => esc_html__( 'Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'selectors' => [
					// The opacity property will override the default inactive dot color which is opacity 0.2.
					'{{WRAPPER}} .swiper-pagination-bullet:not(.swiper-pagination-bullet-active)' => 'background: {{VALUE}}; opacity: 1',
				],
				'condition' => [
					'navigation' => [ 'dots', 'both' ],
				],
			]
		);

		$this->add_control(
			'dots_color',
			[
				'label' => esc_html__( 'Active Color', 'elementor' ),
				'type' => Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .swiper-pagination-bullet' => 'background: {{VALUE}};',
				],
				'condition' => [
					'navigation' => [ 'dots', 'both' ],
				],
			]
		);

		$this->end_controls_section();

	}

    public function dynamic_render($render) {
		ob_start();

		$settings = $this->get_settings_for_display();

        $render = str_replace('settings.', '', $render);
        
		foreach ($settings as $key => $setting) {
			${$key} = $setting;
		}

		try {
		    eval("?> $render <?php ");;
		} catch (ParseError $e) {
		    echo 'Dynamic Widget Debug: ' . $e->getMessage();
		}

		return ob_get_clean();
	}

	public function render_icon_item($item) {
		ob_start();

		$migration_allowed = Elementor\Icons_Manager::is_migration_allowed();

		if ( $item['selected_icon_source'] === 'icon' ) {
			if (! empty( $item['selected_icon']['value'] )) :
				?>
				<span class="softlite-dynamic-icon">
					<?php
						Elementor\Icons_Manager::render_icon( $item['selected_icon'], [ 'aria-hidden' => 'true' ] );
					?>
				</span>
			<?php endif;
		}

		return ob_get_clean();
	}

	private function render_swiper_button( $type ) {
		ob_start();

		$direction = 'next' === $type ? 'right' : 'left';
		$icon_settings = $this->get_settings_for_display( 'navigation_' . $type . '_icon' );

		if ( empty( $icon_settings['value'] ) ) {
			$icon_settings = [
				'library' => 'eicons',
				'value' => 'eicon-chevron-' . $direction,
			];
		}

		Elementor\Icons_Manager::render_icon( $icon_settings, [ 'aria-hidden' => 'true' ] );

		return ob_get_clean();
	}

	protected function render(): void {
		$settings = $this->get_settings_for_display();

		if ( empty( $settings['dynamic_template'] ) ) {
			return;
		}

		foreach ($settings['dynamic_list'] as $key => $item) {
			$settings['dynamic_list'][$key]['icon_html'] = $this->render_icon_item($item);
		}

		if ( !empty($settings['navigation_previous_icon'])) {
			$settings['icon_prev_html'] = $this->render_swiper_button('previous');
		}

		if ( !empty($settings['navigation_next_icon'])) {
			$settings['icon_next_html'] = $this->render_swiper_button('next');
		}

		$slider_settings = [
			'enable' => ! empty( $settings['slider'] ) && $settings['slider'] === 'yes' ? true : false,
			'swiperOptions' => [],
		];

		$active_breakpoints = \Elementor\Plugin::$instance->breakpoints->get_breakpoints();
		$breakpoints = [];
		$breakpoints_point = [0];
		$index = 0;
		foreach ($active_breakpoints as $key => $breakpoint) {
			$breakpoints[$breakpoints_point[$index]] = [];

			if (!empty($settings['slider_slides_to_show_' . $key])) {
				$breakpoints[$breakpoints_point[$index]]['slidesPerView'] = $settings['slider_slides_to_show_' . $key];
			}
			if (!empty($settings['slider_slides_to_scroll_' . $key])) {
				$breakpoints[$breakpoints_point[$index]]['slidesPerGroup'] = $settings['slider_slides_to_scroll_' . $key];
			}
			if (!empty($settings['column_gap_' . $key])) {
				$breakpoints[$breakpoints_point[$index]]['spaceBetween'] = $settings['column_gap_' . $key]['size'];
			}

			$breakpoints_point[] = $breakpoint->get_value();
			$index++;
		}

		$breakpoints[$breakpoints_point[$index]] = [
			'slidesPerView' => $settings['slider_slides_to_show'],
			'slidesPerGroup' => $settings['slider_slides_to_scroll'],
			'spaceBetween' => $settings['column_gap'] ? $settings['column_gap']['size'] : 30,
		];

		$breakpoints = array_reverse($breakpoints, true);
		$breakpoints_point = array_reverse($breakpoints_point);
		$index = 0;

		foreach ($breakpoints as $breakpoint => $breakpoint_settings) {
			if (empty($breakpoint_settings['slidesPerView']) && $index > 0) {
				$breakpoints[$breakpoint]['slidesPerView'] = $breakpoints[strval($breakpoints_point[$index - 1])]['slidesPerView'];
			}
			if (empty($breakpoint_settings['slidesPerGroup']) && $index > 0) {
				$breakpoints[$breakpoint]['slidesPerGroup'] = $breakpoints[strval($breakpoints_point[$index - 1])]['slidesPerGroup'];
			}
			if (empty($breakpoint_settings['spaceBetween']) && $index > 0) {
				$breakpoints[$breakpoint]['spaceBetween'] = $breakpoints[strval($breakpoints_point[$index - 1])]['spaceBetween'];
			}
			$index++;
		}

		$slider_settings['swiperOptions']['breakpoints'] = $breakpoints;
		if (!empty($settings['navigation']) && $settings['navigation'] !== 'none') {
			if (in_array($settings['navigation'], ['dots', 'both'])) {
				$slider_settings['swiperOptions']['pagination'] = [
					'el' => '.swiper-pagination',
					'clickable' => true,
				];
			}
			if (in_array($settings['navigation'], ['arrows', 'both'])) {
				$slider_settings['swiperOptions']['navigation'] = [
					'nextEl' => '.elementor-swiper-button-next',
					'prevEl' => '.elementor-swiper-button-prev',
				];
			}
		}
		if (!empty($settings['infinite'])) {
			$slider_settings['swiperOptions']['loop'] = true;
		}
		if (!empty($settings['autoplay'])) {
			$slider_settings['swiperOptions']['autoplay'] = true;
			if (!empty($settings['autoplay_speed'])) {
				$slider_settings['swiperOptions']['autoplay'] = [];
				$slider_settings['swiperOptions']['autoplay']['delay'] = $settings['autoplay_speed'];
			}
		}
		if (!empty($settings['pause_on_interaction'])) {
			$slider_settings['swiperOptions']['pauseOnMouseEnter'] = true;
		}

		$settings['slider_settings'] = json_encode($slider_settings);

        $loader = new \Twig\Loader\ArrayLoader([]);
        $twig = new \Twig\Environment($loader, [
            'debug' => true,
        ]);
        $twig->addExtension(new \Twig\Extension\DebugExtension());
        $render_template = $this->dynamic_render(str_replace('&&', 'and', $settings['dynamic_template']));

        if (stripos($render_template, 'Dynamic Widget Debug') === false) {
            try {
                $twig->tokenize(str_replace('&&', 'and', $settings['dynamic_template']));
                $template = $twig->createTemplate($render_template);
                echo '<div class="softlite-wrapper">' . $template->render($settings) . '</div>';
            } catch (\Twig_Error_Syntax $e) {
                $error_message = $e->getMessage();
                $error_message = explode('"', $error_message);

                echo '<pre>';
                echo 'Dynamic Widget Debug: ' . $error_message[0];
                echo '</pre>';
            }
        } else {
            echo '<pre>';
            echo $render_template;
            echo '</pre>';
        }
	}

	protected function content_template(): void {
		?>
		<#
            if ( '' === settings.dynamic_template ) {
                return;
            }
            
            function twigToTemplateLiteral(twigTemplate) {
				return twigTemplate
					// First handle if/else
					.replace(/{%\s*if\s+([^%]+)\s*%}((?:(?!{%\s*(?:if|endif|else)\s*%}).)*){%\s*else\s*%}((?:(?!{%\s*(?:if|endif)\s*%}).)*){%\s*endif\s*%}/g, '${$1 ? `$2` : `$3`}')
					// Then handle simple if
					.replace(/{%\s*if\s+([^%]+)\s*%}((?:(?!{%\s*(?:if|endif)\s*%}).)*){%\s*endif\s*%}/g, '${$1 ? `$2` : ``}')
					// Handle for loops
					.replace(/{%\s*for\s+(\w+)\s+in\s+([^\s%]+)\s*%}([\s\S]*?){%\s*endfor\s*%}/g, '${$2.map(($1) => `$3`).join(\'\')}')
					// Handle raw filter
					.replace(/\{\{\s*([^}]+?)\|raw\s*\}\}/g, '${$1}')
					// Handle regular variables
					.replace(/{{\s*([^}]+)\s*}}/g, '${$1}');
			}

			// loop through dynamic_list and render icon
			settings.dynamic_list?.forEach((item, index) => {
				const iconHTML = elementor.helpers.renderIcon( view, item.selected_icon, { 'aria-hidden': true }, 'i' , 'object' );
				settings.dynamic_list[index].icon_html = iconHTML ? iconHTML.value : '';
			});

			const activeBreakpoints = elementor.config.responsive.activeBreakpoints;

			let breakpoints = {};
			let breakpointsPoint = [0];
			let index = 0;

			// Duyệt qua các breakpoint
			Object.keys(activeBreakpoints).forEach((breakpointName) => {
				const breakpoint = activeBreakpoints[breakpointName];

				if (settings[`slider_slides_to_show_${breakpointName}`]) {
					if (!breakpoints[breakpointsPoint[index]]) breakpoints[breakpointsPoint[index]] = {};
					breakpoints[breakpointsPoint[index]].slidesPerView = settings[`slider_slides_to_show_${breakpointName}`];
				}

				if (settings[`slider_slides_to_scroll_${breakpointName}`]) {
					if (!breakpoints[breakpointsPoint[index]]) breakpoints[breakpointsPoint[index]] = {};
					breakpoints[breakpointsPoint[index]].slidesPerGroup = settings[`slider_slides_to_scroll_${breakpointName}`];
				}

				if (settings[`column_gap_${breakpointName}`]) {
					if (!breakpoints[breakpointsPoint[index]]) breakpoints[breakpointsPoint[index]] = {};
					breakpoints[breakpointsPoint[index]].spaceBetween = settings[`column_gap_${breakpointName}`].size;
				}

				breakpointsPoint.push(breakpoint.value);
				index++;
			});

			breakpoints[breakpointsPoint[index]] = {
				slidesPerView: settings.slider_slides_to_show,
				slidesPerGroup: settings.slider_slides_to_scroll,
				spaceBetween: settings.column_gap ? settings.column_gap.size : 30,
			};

			// Reverse breakpoints and breakpointsPoint
			breakpoints = Object.keys(breakpoints)
				.sort((a, b) => b - a)
				.reduce((acc, key) => {
					acc[key] = breakpoints[key];
					return acc;
				}, {});

			breakpointsPoint = breakpointsPoint.reverse();
			index = 0;

			Object.keys(breakpoints).forEach((breakpoint) => {
				const prevIndex = index - 1;
				if (!breakpoints[breakpoint].slidesPerView && prevIndex >= 0) {
					breakpoints[breakpoint].slidesPerView = breakpoints[breakpointsPoint[prevIndex]].slidesPerView;
				}
				if (!breakpoints[breakpoint].slidesPerGroup && prevIndex >= 0) {
					breakpoints[breakpoint].slidesPerGroup = breakpoints[breakpointsPoint[prevIndex]].slidesPerGroup;
				}
				if (!breakpoints[breakpoint].spaceBetween && prevIndex >= 0) {
					breakpoints[breakpoint].spaceBetween = breakpoints[breakpointsPoint[prevIndex]].spaceBetween;
				}
				index++;
			});

			const swiperOptions = {
				breakpoints: breakpoints,
			};

			if (settings.navigation !== 'none') {
				if (settings.navigation === 'dots' || settings.navigation === 'both') {
					swiperOptions.pagination = {
						el: '.swiper-pagination',
						clickable: true,
					};
				}

				if (settings.navigation === 'arrows' || settings.navigation === 'both') {
					swiperOptions.navigation = {
						nextEl: '.elementor-swiper-button-next',
						prevEl: '.elementor-swiper-button-prev',
					};

					const icon_prev_html = elementor.helpers.renderIcon( view, settings.navigation_previous_icon, { 'aria-hidden': true }, 'i' , 'object' );
					const icon_next_html = elementor.helpers.renderIcon( view, settings.navigation_next_icon, { 'aria-hidden': true }, 'i' , 'object' );

					settings.icon_prev_html = icon_prev_html.value ? icon_prev_html.value : '<i aria-hidden="true" class="eicon-chevron-left"></i>';
					settings.icon_next_html = icon_next_html.value ? icon_next_html.value : '<i aria-hidden="true" class="eicon-chevron-right"></i>';
				} else {
					settings.icon_prev_html = '';
					settings.icon_next_html = '';
				}
			}  else {
				settings.icon_prev_html = '';
				settings.icon_next_html = '';
			}

			if (settings.autoplay === 'yes') {
				swiperOptions.autoplay = true;
				if (settings.autoplay_speed) {
					swiperOptions.autoplay = {
						delay: settings.slider_autoplay_speed,
					};
				}
			}

			if (settings.infinite === 'yes') {
				swiperOptions.loop = true;
			}

			if (settings.pause_on_interaction === 'yes') {
				swiperOptions.pauseOnMouseEnter = true;
			}

			settings.slider_settings = JSON.stringify({
				enable: settings.slider === 'yes',
				swiperOptions: swiperOptions,
			}).replace(/"/g, "'");
            
            let templateHtml = '';

            try {
                const template = twigToTemplateLiteral(settings.dynamic_template);
                templateHtml = new Function('settings', `return \`${template}\`;`)(settings);
            } catch (error) {
                templateHtml = 'Syntax Error: ' + error.message;
            }
		#>
            <div class="softlite-wrapper">{{{ templateHtml }}}</div>
		<?php
	}
}
